<?php /** @noinspection DuplicatedCode */


namespace ifes;

use Appointment;
use BackButton;
use DateInterval;
use DateTime;
use Exception;
use HangupButton;
use CSS;
use JS;
use NextButton;
use Progressbar;
use ProjectInfo;
use SurveyInfo;
use TestInfo;
use TStatus;

class IfesDesignV2
{

    const IFES_DESIGN_NAME = 'IFES_responsive';
    const REGEX_TAG = '/(?:<tr>)?<td[^>]*id="([^"]*)"[^>]*paneltype=(\d+)(?:[^>](?!display))*(?:;?display\s*:\s*(ifes-[^;]*);)?[^>]*>(.*)/is';
    const REGEX_STYLE = '/\sstyle=["\'][^"\']*["\']/i';
    const REGEX_ANIMATE_STYLESHEET = '/<link rel="stylesheet" href="..\/_css\/animate.css">/i';
    const REGEX_LABEL_VALUES = '/^([^:]+):(.*)$/i';
    const HTML_PARSE_ERROR_CSS_CLASS = 'ifes-error-html-parser';
    const LAYOUT_PANELTYPES = [3, 4, 6, 40];
    const KNOWN_PANELTYPES = [
        3 => 'question',
        4 => 'instruction',
        6 => 'answer',
        7 => 'variable',
        11 => 'back',
        15 => 'progressbar',
        19 => 'text',
        10 => 'submit',
        24 => 'hangup',
        25 => 'cancel-interview',
        26 => 'break-goto-idle',
        27 => 'quota',
        28 => 'interviewer-id',
        29 => 'project-info',
        30 => 'tel-structure-data',
        31 => 'call-status',
        33 => 'appointment',
        35 => 'dialer',
        38 => 'phone-number',
        39 => 'caller-name',
        40 => 'error',
        41 => 'go-to-idle',
        42 => 'make-appointment',
        46 => 'appointment-comment',
        54 => 'appointment-call-time',
        57 => 'script',
        63 => 'interviewer-username'
    ];

    const KNOWN_QUESTION_TYPES = [
        1 => 'single-response',
        2 => 'scale',
        3 => 'multi-response',
        4 => 'open',
        5 => 'instruction',
        6 => 'time-stamp',
        12 => 'recording',
        13 => 'captcha',
        14 => 'gps'
    ];

    private $questionScriptOutput = "";

    public function writeQuestionScriptOutput()
    {
        echo $this->questionScriptOutput;
    }

    public function writePageHeader()
    {
        echo HtmlPageHeader::render();
    }

    public function writeFormWrapperStartTag()
    {
        global $db;
        global $questionnaire;
        $templateName = trim((string)$db->get(DB_WARPIT_WEBCATI_BASE . "._DesignerTemplates", $questionnaire->display->getIdTemplate(), "name"));
        $css[] = "ifes-responsive";
        $css[] = strtolower(str_replace('_', '-', str_replace(' ', '', $templateName)));
        $css[] = SurveyInfo::isMobileDevice() ? "mobile" : "desktop";
        echo '<div class="' . $this->getCssString($css) . '">';

        if (SurveyInfo::isTest() && TestInfo::isEnabled()) {
            echo "<div class=\"ifes-test-interview-warning\">" . TestInfo::getMessage() . "</div>";
        }
    }

    public function writeFormStartTag() {
        echo "<form id='mainForm' method='POST' action='" . htmlspecialchars($_SERVER['PHP_SELF']) . "'>";
    }

    public function writeFormEndTag()
    {
        echo '</form>';
    }

    public function writeFormWrapperEndTag()
    {
        echo '</div>';
    }

    /**
     * @param int $questionId
     * @return void
     */
    public function executeScriptsForQuestion($questionId)
    {
        global $db;
        // reversed order of script execution to allow question specific overrides
        // 1: beforeEveryQuestion
        // 2: beforeDisplay
        $sql = "SELECT script FROM " . DB_WARPIT_WEBCATI . " ." . ProjectInfo::name() . "_script " .
            "WHERE (id_question=" . $questionId . " AND id_event = 2) OR (id_question = 0 AND id_event = 3) " .
            "ORDER BY id_event DESC";
        $res = $db->SQLexecute($sql);

        ob_start();
        while ($row = $db->fetchRow($res)) {
            $script = $row[0];
            $script .= "\nif (false) {}"; // avoid parse error, if script only contains comments
            eval($script);
        }

        $this->questionScriptOutput = ob_get_contents();
        ob_end_clean();
    }

    public function executePHPIncludeScript()
    {
        global $db;
        $lastError = error_get_last();
        ob_start();
        $sql = "SELECT script FROM " . DB_WARPIT_WEBCATI . "." . ProjectInfo::name() . "_script WHERE id_question = 0 AND id_event = 100001";
        $res = $db->SQLexecute($sql);

        while ($row = $db->fetchRow($res)) {
            $script = $row[0];
            $script .= "\nif (false) {}"; // avoid parse error, if script only contains comments
            eval($script);
        }
        $scriptOutput = ob_get_clean();
        ob_end_clean();

        if ($lastError != error_get_last()) {
            echo "<h1>Error executing script 'Project PHP Functions'</h1>";
            die($scriptOutput);
        }
    }

    public function writeCssLinks()
    {
        foreach (CSS::getFiles() as $css) {
            echo "    <link rel=\"stylesheet\" href=\"$css\" type=\"text/css\" />\n";
        }
    }

    public function writeProjectCss()
    {
        global $db;
        $css = $db->get(DB_WARPIT_WEBCATI . " ." . $_SESSION['proj_name'] . "_script", "id_question=0 AND id_event = " . DesignConstants::SCRIPT_ID_CSS, "script");
        if (is_array($css)) {
            echo "<style>\n";
            echo $css[0];
            echo "\n</style>\n";
        }
    }

    public function writeJsHeadScripts()
    {
        foreach (JS::getHeadFiles() as $js) {
            echo "    <script type=\"text/javascript\" src=\"$js\"></script>\n";
        }
    }

    public function writeJsBottomScripts()
    {
        foreach (JS::getBottomFiles() as $js) {
            echo "<script type=\"text/javascript\" src=\"$js\"></script>\n";
        }
    }

    private function getCssString($css) {
        if (!$css) return '';
        if (is_array($css)) {
            $cssString = '';
            foreach ($css as $cls) {
                $cls = trim($cls);
                if (!$cls) continue;
                $cssString .= " $cls";
            }
            return trim($cssString);
        }
        return trim($css);
    }

    private function setDesignerTexts()
    {
        global $designerText;
        $designerText->setUserDefinedText(DesignConstants::TEMPLATE_VAR_BACK_BTN_TEXT, BackButton::getText());
        $designerText->setUserDefinedText(DesignConstants::TEMPLATE_VAR_BACK_BTN_VISIBILITY, BackButton::isVisible() ? 'block' : 'hidden'); //TODO needed?
        $designerText->setUserDefinedText(DesignConstants::TEMPLATE_VAR_SUBMIT_BTN_TEXT, NextButton::getText());
    }

    public function processXmlTemplate($display, $xmlObj, $parType, $lastChild, $parId = null)
    {
        global $qVar;
        $this->setDesignerTexts();
        $attr = $xmlObj->attributes();
        $paneltype = $attr['type'];

        // 1 = horizontal, 2 = vertical
        if ($paneltype == 1 || $paneltype == 2) {
            $css = $attr[DesignConstants::XML_TEMPLATE_CSS_CLASS_PROPERTY];
            if ($css) $css = ' class="' . $this->getCssString($css) . '"';
            $html = "<div $css>";
            foreach ($xmlObj->children() as $child) {
                $html .= $this->processXmlTemplate($display, $child, $parType, $lastChild, $parId);
            }
            $html .= '</div>';
            return $html;
        } else if ($paneltype == 47) { // template
            global $db;
            $templateName = null;
            $sql = "SELECT name FROM " . DB_WARPIT_WEBCATI_BASE . "._DesignerTemplates WHERE id = " . $attr['id_template'];
            if (!$rs = $db->SQLexecute($sql)) Log::error($db->GetError());
            if ($row = $db->fetchRow($rs)) {
                $templateName = $row[0];
            }
            $html = $display->displayChilds($xmlObj, $parType, $lastChild, $parId);
            $css[] = strtolower(str_replace('_', '-', $templateName));
            if ($qVar['q_name']) {
                $qName = $qVar['q_name'];
                $qName = str_replace('_', '-', $qName);
                if (strpos($qName, '-') === 0) {
                    $qName = substr($qName, 1);
                }
                $css[] = 'q-' . strtolower($qName);
            }
            return $this->processHtml($html, $css);
        } else {
            return $this->processHtml($display->displayChilds($xmlObj, $parType, $lastChild, $parId));
        }
    }

    public function processHtml($html, $css = array())
    {
        $pHtml = trim($html);
        if ($pHtml == '') return '';
        if (preg_match(self::REGEX_TAG, $pHtml, $matches)) {
            $elementId = $matches[1];
            $paneltype = $matches[2];

            $css[] = $this->getCssClass($paneltype);
            $pHtml = $this->processPanel($paneltype, $matches[4]);

            if ($pHtml === false) {
                return $this->htmlParseError($html, $paneltype);
            } elseif (empty($pHtml)) { //TODO keep this?
                return '';
            } else {
                /** @noinspection HtmlUnknownAttribute */ //TODO keep CSS class ifes-panel?
                return "\n<div id=\"" . $elementId . '" paneltype="' . $paneltype . '" class="' . $this->getCssString($css) . "\">" . $pHtml . "</div>\n";
            }
        } else {
            return $this->htmlParseError($html);
        }
    }

    private function processPanel($paneltype, $html)
    {
        global $qVar;
        $questionName = $qVar['q_name'];

        $html = trim($html);

        if ($html == '') return '';

        switch ($paneltype) {
            case 3: // question
            case 4: // instruction
            case 19: // textbox
            case 26: // break-goto-idle
            case 27: // quota
            case 35: // dialer
            case 40: // validation error
            case 57: // script
                return $html;

            case 6: // answers
                return preg_replace(self::REGEX_ANIMATE_STYLESHEET, '', $html); //TODO remove hidden inputs?

            case 7: // variable
                return "<span>Frage: <b>$html</b></span>";

            case 47: // question template
                $html = trim(preg_replace(self::REGEX_STYLE, '', $html));
                if (str_contains($html, "class='answersTable'")) {
                    return '<p>fix processing of paneltype 47</p>';
                }
                $panels = explode('<tr>', $html);
                $html = '';
                foreach($panels as $panel) {
                    $panel = trim($panel);
                    if (!starts_with($panel, '<td')) continue;
                    $panel = "<tr>$panel";
                    $html .= $this->processHtml($panel);
                }
                return $html;

            case 10: // submit
                $html = '<button class="btn ifes-btn ifes-btn-submit" 
                                 type="submit" 
                                 name="questSubmit" 
                                 id="submitCallStatus" 
                                 value="submit" 
                                 onclick="recodeClickEvents(this,3);hideButton(\'submitCallStatus\');">';
                $html .= NextButton::getText();
                $html .= '</button>';
                if (ProjectInfo::submitOnEnter()) {
                    $html .= "<script> document.addEventListener('keypress', (event)=>{ if (event.key === 'Enter') $('#submitCallStatus').click(); });</script>";
                }
                return $html;

            case 11: // back
                $html = '<button class="btn ifes-btn ifes-btn-back" 
                                 type="submit" 
                                 name="questBack" 
                                 id="questBack" 
                                 value=" " 
                                 onclick="recodeClickEvents(this,3);hideButton(\'questBack\'); return false;">';
                $html .= BackButton::getText();
                $html .= '</button>';
                return $html;

            case 15:
                return $this->renderProgressbar();

            case 25: // cancel interview
            case 42: // make appointment
                if (!preg_match('/(?:(<input.*class\s*=\s*["\'][^"\']*["\'])(.*)|(<input))(.*)/is', $html, $matches)) return false;
                $css = (array_key_exists($paneltype, self::KNOWN_PANELTYPES) ? self::KNOWN_PANELTYPES[$paneltype] : 'unknown');
                $css = "btn ifes-btn ifes-btn-$css";
                $disabled = '';
                if ($paneltype == 11 && !BackButton::isVisible()) {
                    return '';
                } elseif ($paneltype == 42 && SurveyInfo::isTest()) {
                    $disabled = 'disabled';
                }
                if ($matches[1]) {
                    $html = "$matches[1] $disabled $css $matches[2]";
                    $html = "<input $disabled class=\"$css\"$matches[2]";
                } else {
                    $html = "$matches[3] $disabled class=\"$css\"$matches[4]";
                }
                $html = str_replace('Are you sure?', 'Sind Sie sicher?', $html); //TODO configure warpit correct?
                if ($paneltype == 10 && $questionName == 'tStatus' && TStatus::isSubmitOnEnterEnabled()) {
                    //TODO test
                    $html .= "<script> document.addEventListener('keypress', (event)=>{ if (event.key === 'Enter') $('#submitCallStatus').click(); });</script>";
                }
                return $html;

            case 24: //hangup
                if (!HangupButton::isVisible()) return '';
                return '<button type="button" class="ifes-btn ifes-btn-hangup" name="hangupButton" id="hangupButton">' . HangupButton::getText() . '</button>';

            case 28: // interviewer id
                if (!preg_match(self::REGEX_LABEL_VALUES, $html, $matches)) return false;
                return $this->getLabelValueHtml($paneltype, ["Interviewer ID" => $matches[2]], false);

            case 29: // project info
                return ''; //TODO project info
                $config = $this->config->tStatus()->projectInfo();
                /** @noinspection DuplicatedCode */
                $lines = explode('<br>', $html);
                $items = array();
                foreach ($lines as $line) {
                    if (trim($line) == '') continue;
                    if (!preg_match(self::REGEX_LABEL_VALUES, $line, $matches)) return false;
                    $label = trim($matches[1]);
                    $value = trim($matches[2]);
                    $items[$label] = $value;
                }
                $items = $config->process($items);
                return empty($items) ? '' : $this->getLabelValueHtml($paneltype, $items, false);

            case 30: // tel structure data
                return ''; //TODO tel structure data panel
                $config = $this->config->tStatus()->telInfo();
                /** @noinspection DuplicatedCode */
                $lines = explode('<br>', $html);
                $items = array();
                foreach ($lines as $line) {
                    if (trim($line) == '') continue;
                    if (!preg_match(self::REGEX_LABEL_VALUES, $line, $matches)) return false;
                    $label = trim($matches[1]);
                    $value = trim($matches[2]);
                    $items[$label] = $value;
                }
                $items = $config->process($items);
                return empty($items) ? '' : $this->getLabelValueHtml($paneltype, $items, false);

            case 31: // call status
                $scriptStart = strpos($html, '<script');
                if ($scriptStart === false) {
                    $script = "";
                } else {
                    $script = substr($html, $scriptStart);
                    $html = substr($html, 0, $scriptStart - 1);
                }
                $html = trim(str_replace(["\n", "\r", "\r\n", "\n\r"], '', $html));
                $inputTags = explode('<br>', $html);
                unset($inputTags[0]);

                $oTag = '<div class="ifes-input-wrapper">';
                $header = '<div class="ifes-p-call-status-header">' . TStatus::getCallStatusHeaderText();
                if (TStatus::getCallStatusHelpText()) {
                    $header .= '<span class="ifes-help ifes-call-status-help">' . TStatus::getCallStatusHelpText() . '</span>';
                }
                $header .= '</div>';
                $items = '<div class="ifes-p-call-status-items"><div class="ifes-input-wrapper">' . implode('</div><div class="ifes-input-wrapper">', $inputTags) . '</div></div>';
                return $header . '<div class="ifes-p-call-status-items">' . $oTag . implode('</div>' . $oTag, $inputTags) . $script . '</div></div>';

            case 33: // appointment
                $now = new DateTime('now');
                $normalizedNow = new DateTime('now');

                $dpStep = Appointment::getTimePickerInterval();
                $dpMinTime = Appointment::getMinTime();
                $dpMaxTime = Appointment::getMaxTime();
                $dpMaxDate = Appointment::getMaxDate();
                $dpMinDate = $normalizedNow->format('d-m-Y H:i');

                $delta = $dpStep - $normalizedNow->format('i') % $dpStep;
                $normalizedNow->add(new DateInterval('PT' . $delta . 'M'));
                $warpitDate = $now->format('d-m-Y');
                $deDate = $now->format('d.m.Y');
                $dpDate = $now->format('Y/m/d');
                $time = $now->format('H:i');
                $normalizedTime = $normalizedNow->format('H:i');

                $phonenumberPlaceholder = Appointment::getPhonenumberPlaceholder();
                $phonenumber = '';
                if ($_SESSION['app_call_this_num'])
                    $phonenumber = join('', explode('/', $_SESSION['app_call_this_num']));
                $comment = strlen($_SESSION['app_koment']) > 0 ? $_SESSION['app_koment'] : $_SESSION['app_display_komment'];

                $submitOnEnterJS = '';
                if (TStatus::isSubmitOnEnterEnabled()) {
                    $submitOnEnterJS = "document.addEventListener('keypress', (event)=>{ if (event.target.id !== 'dogovor_komentar' && event.key === 'Enter') $('#submitCallStatus').click(); });";
                }
                /** @noinspection BadExpressionStatementJS */
                /** @noinspection JSUnusedLocalSymbols */
                /** @noinspection ReservedWordAsName */
                /** @noinspection JSUnnecessarySemicolon */
                return <<<EOF
                    <script>
                        function warpitClientExecuteOnLoad() {
                            const errorMsg = $('#ifesErrorMsg');
                            const datetimepickerValue = $('#datetimepickerValue');
                            const datetimepickerHolder = $('#datetimepickerHolder');
                            const dateInput = $('#dateInput');
                            const phoneNrInput = $('#app_call_this_num');  

                            const validateAndSyncDate = function(date, updateDatePicker) {
                                dateInput.val(date.format('DD.MM.YYYY HH:mm'));

                                if (!date.isValid()) {
                                    errorMsg.text('Datum für Termin ist ungültig.').show();
                                    return false;
                                } else if (date < moment('$warpitDate $time', 'DD-MM-YYYY HH:mm')) {
                                    errorMsg.text('Der Termin liegt in der Vergangenheit.').show();
                                    return false;
                                } else if (date.hour() < $dpMinTime || date.hour() > $dpMaxTime || (date.hour() === $dpMaxTime && date.minute() > 0)) {
                                    errorMsg.text('Termine können nur zwischen {$dpMinTime}h und {$dpMaxTime}h vereinbart werden.').show();
                                    return false;
                                } else {
                                    // fix minutes to match step
                                    /*
                                    let delta = date.minutes() % $dpStep;
                                    delta = delta < ($dpStep / 2) ? delta * -1 : $dpStep - delta;
                                    let dpDate = date.add(delta, 'minutes');
                                     */

                                    dateInput.val(date.format('DD.MM.YYYY HH:mm'));
                                    datetimepickerValue.val(date.format('DD-MM-YYYY HH:mm'));
                                    
                                    if (updateDatePicker) {
                                        datetimepickerHolder.val(date.format('YYYY/MM/DD HH:mm'));
                                        datetimepickerHolder.blur();

                                        setTimeout(function() {
                                            let delta = date.minutes() % $dpStep;
                                            delta = delta < ($dpStep / 2) ? delta * -1 : $dpStep - delta;
                                            let dpDate = date.add(delta, 'minutes');
                                            const hours = dpDate.hours();
                                            const minutes = dpDate.minutes();
                                            const timeSlotCount = $('.xdsoft_time').length;
                                            const currentTimeSlot = $('.xdsoft_time.xdsoft_current');
                                            const currentTimeSlotIndex = currentTimeSlot.index();
                                            currentTimeSlot.removeClass('xdsoft_current');
                                            const selector = ".xdsoft_time[data-hour='" + hours + "'][data-minute='" + minutes + "']";
                                            $(selector).addClass('xdsoft_current');
                                            const timeSlots = $('.xdsoft_time_variant');
                                            let offset = 3;
                                            if (currentTimeSlotIndex > timeSlotCount - 6) offset = 6;
                                            let timeSlotsMargin = -1 * (currentTimeSlotIndex - offset) * timeSlots.height() / timeSlotCount;
                                            if (timeSlotsMargin > 0) timeSlotsMargin = 0;

                                            const timebox = $('.xdsoft_time_box');
                                            const scrollbar = timebox.find('.xdsoft_scroller');
                                            const scrollbarMargin = (timebox.find('.xdsoft_scrollbar').height() - scrollbar.height()) / timeSlotCount * currentTimeSlotIndex;

                                            scrollbar.css('margin-top', scrollbarMargin);
                                            timeSlots.css('margin-top', timeSlotsMargin);
                                        }, 100);
                                
                                    }
                                    errorMsg.text('').hide();
                                    return true;
                                }
                            }

                            const validatePhoneNr = function() {
                                const nr = phoneNrInput.val().replace(/[\/-\s]/g, '');
                                if (nr !== '') {
                                    if (isNaN(+nr) || isNaN(parseInt(nr))) {
                                        errorMsg.text('Telefonnummer muss numerisch sein.').show();
                                        return false;
                                    } else if (!nr.startsWith('0')) {
                                        errorMsg.text('Telefonnummer muss mit 0 beginnen.').show();
                                        return false;
                                    }
                                }
                                phoneNrInput.val(nr);
                                errorMsg.text('').hide();
                                return true;
                            }

                            datetimepickerHolder.datetimepicker({
                                inline: true,
                                dayOfWeekStart: 1, 
                                onChangeDateTime: function(date) {
                                    validateAndSyncDate(moment(date), false);
                                },
                                minDate: '$dpMinDate',
                                maxDate: '$dpMaxDate',
                                minTime: '$dpMinTime:00',
                                maxTime: '$dpMaxTime:01',
                                lang: 'de',
                                step: $dpStep
                            });
                            dateInput.on('change', function() {
                                validateAndSyncDate(moment($(this).val(), 'DD.MM.YYYY HH:mm'), true);
                            });

                            $('#submitCallStatus').on('click', function(event) {
                                const date = moment(dateInput.val(), 'DD.MM.YYYY HH:mm');
                                if (!validateAndSyncDate(date) || !validatePhoneNr()) {
                                    event.preventDefault();
                                }
                            });

                            phoneNrInput.on('change', function() {
                                validatePhoneNr();
                            });
                            
                            $submitOnEnterJS
                        }
                    </script>
                    <input type="hidden" value="$warpitDate $normalizedTime" name="appDateTimeValue" id="datetimepickerValue">
                    <div class="ifes-error-msg" id="ifesErrorMsg" style="display: none;"></div>
                    <div class="ifes-make-appointment">
                        <div id="datetimepicker"><input type="text" value="$dpDate $normalizedTime" id="datetimepickerHolder"></div>
                        <label for="dateInput">Termin:</label>
                        <input type="text" value="$deDate $normalizedTime" name="appDateTimeDisplay" id="dateInput">
                        <label for="dogovor_komentar">Kommentar:</label>
                        <textarea onkeydown="arguments[0].stopPropagation()" name="dogovor_komentar" id="dogovor_komentar">$comment</textarea>
                        <label for="app_call_this_num">Neue Telefonnummer:</label>
                        <input type="text" name="app_call_this_num" maxlength="20" id="app_call_this_num" placeholder="$phonenumberPlaceholder" value="$phonenumber">
                        <!--span class="ifes-help ifes-make-appointment-phone-number-help">(z.B.: 06641234567)</-span-->
                    </div>
EOF;

            case 38: // phone number
                $phonenumber = null;
                $id = null;
                if (preg_match('/<span\sid\s*=\s*["\']*([^"\'\s]+)["\']*\s*><a[^>]*>([^<]*)<.*/is', $html, $matches)) {
                    $id = trim($matches[1]);
                    $phonenumber = trim($matches[2]);
                } else if (preg_match(self::REGEX_LABEL_VALUES, $html, $matches)) {
                    $phonenumber = trim($matches[2]);
                }

                if (!$phonenumber) return false;
                if ($id) $id = ' id="' . $id . '"';
                $helpText = TStatus::getPhoneNumberHelpText();
                if ($helpText) $helpText = '<span class="ifes-help ifes-phone-number-help">' . htmlspecialchars($helpText) . '</span>';

                return '<span' . $id . '><a href="callto:' . str_replace('/', '', $phonenumber) . '">Tel: ' . $phonenumber . '</a></span>' . $helpText;

            case 39: // caller name TODO
                if ($questionName == 'tStatus' && !TStatus::isCallerNameEnabled()) return '';
                if (!preg_match(self::REGEX_LABEL_VALUES, $html, $matches)) return false;
                $name = trim($matches[2]);
                if (!$name || $name == '-') return '';
                return "<span>$name</span>";

            case 41: // gotoIdle checkbox
                if (!preg_match('/^<.*value=[\'"]*([0-9]*)[^>]*name=["\']*([^\'"]*).*>([^<]*)<br>(.*)$/is', $html, $matches)) return false;
                $value = trim($matches[1]);
                $name = trim($matches[2]);
                $text = trim($matches[3]);
                $select = trim($matches[4]);
                $checked = '';
                if ($_SESSION['gotoIdle']) $checked = 'checked';
                return "<div class=\"ifes-input-wrapper\"><input type=\"checkbox\" id=\"$name\" value=\"$value\" name=\"$name\" $checked><label for=\"$name\">$text</label></div><div>$select</div>";

            case 46: // appointment comment
                if (!preg_match('/^<div[^>]+>(.*)<\/div>\s*$/is', $html, $matches)) return false;
                $comment = trim($matches[1]);
                if (!$comment) return '';
                return "<div>$comment</div>";

            case 54: // appointment call time
                if (!preg_match('/.*([0-9]{4}\.[0-9]{2}\.[0-9]{2}\s[0-9]{2}:[0-9]{2}:[0-9]{2}).*/is', $html, $matches)) return false;
                $appointment = trim($matches[1]);
                if (!$appointment) return '';

                $appointment = date_format(date_create_from_format('Y.m.d H:i:s', $appointment), 'd.m.Y H:i');
                $displayText = "Erneuter Anwahlversuch";
                if ($_SESSION['app_exists']) {
                    $displayText = "Vereinbarter Termin";
                }
                return "<span>" . $displayText . ": " . $appointment . "</span>";

            case 63: // interviewer username
                $username = explode('_', $html)[0];
                return $this->getLabelValueHtml($paneltype, ["Kennung" => $username], false);
        }

        return false;
    }

    private function htmlParseError($html, $paneltype = "-")
    {
        if ($html == '') return '';
        return "\n<div unknown-paneltype=\"$paneltype\" class=\"" . self::HTML_PARSE_ERROR_CSS_CLASS . '"><table>' . $html . "</td></tr></table></div>\n";
    }

    private function getPanelName($paneltype)
    {
        return (array_key_exists($paneltype, self::KNOWN_PANELTYPES) ? self::KNOWN_PANELTYPES[$paneltype] : 'unknown');
    }

    private function getCssClass($paneltype)
    {
        if ($paneltype == 47) return '';
        $css = '';
        if (in_array($paneltype, self::LAYOUT_PANELTYPES)) {
            $css = 'ifes-p ';
        }
        $css .= 'ifes-p-' . $this->getPanelName($paneltype);

        if ($paneltype == 6) { // answer -> add class for answer type
            global $qVar;
            $answerType = self::KNOWN_QUESTION_TYPES[$qVar['tip_sql']];
            if ($answerType) {
                $css .= " ifes-p-$answerType";
            }
        }
        return $css;
    }

    private function getLabelValueHtml($paneltype, $items, $skipEmpty, $id = null)
    {
        $cssClass = $this->getCssClass($paneltype);

        $html = '';
        foreach ($items as $label => $value) {
            $value = trim($value);
            if ($skipEmpty && (!$value || $value == '')) continue;

            $label = trim($label);
            if ($id) $id = 'id="' . trim($id) . '"';
            $html .= '<span class="ifes-label ' . $cssClass . '-label">' . trim($label) . '</span>';
            $html .= '<span ' . $id . ' class="ifes-value ' . $cssClass . '-value">' . trim($value) . '</span>';
        }

        if ($html) $html = '<div class="ifes-listing ' . $cssClass . '-listing">' . $html . '</div>';
        return $html;
    }

    private function renderProgressbar()
    {
        if (!Progressbar::isVisible()) return '';
        $template = trim(Progressbar::getTemplate());
        if (!$template) return '';

        $steps = Progressbar::getProgressSteps();
        if (!empty($steps)) {
            // manual progress
            $stepCount = count($steps) + 1;
            $idCurrentQuestion = explode('/', $_SESSION['currentRotationString'])[1];
            if (!$idCurrentQuestion) $idCurrentQuestion = 0;

            $sql = 'SELECT t1.id_question as id, t1.idLab as qname FROM ';
            $sql .= DB_WARPIT_WEBCATI . '.' . ProjectInfo::treeTableName() . ' t1 LEFT OUTER JOIN ';
            $sql .= DB_WARPIT_WEBCATI . '.' . ProjectInfo::treeTableName() . ' t2 ON t1.id_sup = t2.id ';
            $sql .= 'WHERE t1.is_leaf = 1 AND (t1.id_question = ' . $idCurrentQuestion . " OR t1.idLab in ('" . implode("','", $steps) . "'))";
            $sql .= 'ORDER BY IFNULL(t2.orderBy, t1.orderBy), CASE WHEN ISNULL(t2.orderBy) THEN 0 ELSE t1.orderBy END';

            global $db;
            $res = $db->SQLexecute($sql);

            $currentStep = -1;
            while ($row = $db->fetchAssoc($res)) {
                if (in_array($row['qname'], $steps)) $currentStep++;
                if ($row['id'] == $idCurrentQuestion) {
                    $currentStep++;
                    break;
                }
            }
            if ($currentStep < 0) $currentStep = 0;

            $progress = round($currentStep / $stepCount * 100);
        } else {
            // calculated progress
            global $qVar;
            global $questionnaire;

            if ($questionnaire && $qVar['q_pos'] > 0) {
                $qPositionAndAll = $questionnaire->getQuestionPosition();
                $qPos = $qPositionAndAll[0];
                $qAll = $qPositionAndAll[1];
            } else if (!$questionnaire) //question preview
            {
                $qPos = 4;
                $qAll = 10;
            } else
                return '';
            $progress = round($qPos / $qAll * 100);
            $progress = 50;
        }

        $vars = array(
            '[!progress!]' => $progress . '%',
            '[!progresstext!]' => (Progressbar::isPercentageVisible() ? $progress . '%' : '')
        );
        return strtr($template, $vars);
    }


}
