<?php

use ifes\ConfigStorageArea;
use ifes\FileChecks;

abstract class PageHeader
{
    const ENABLED_KEY = "enabled";
    const LOGO_VISIBLE_KEY = "logo_visible";
    const TITLE_KEY = "title";
    const TEMPLATE_KEY = "template";
    const MENU_KEY = "menu";
    const MENU_HTML_KEY = "menu_html";
    const MENU_VISIBLE_KEY = "menu_visible";

    /* @var $storage ConfigStorageArea */
    private static $storage;

    private static function init()
    {
        if (!self::$storage) {
            self::$storage = new ConfigStorageArea(get_called_class());
        }
    }

    public static function enable() {
        self::init();
        self::$storage->set(self::ENABLED_KEY, true);
    }

    public static function disable() {
        self::init();
        self::$storage->set(self::ENABLED_KEY, false);
    }

    public static function isEnabled() {
        self::init();
        return self::$storage->get(self::ENABLED_KEY);
    }

    public static function showLogo() {
        self::init();
        self::$storage->set(self::LOGO_VISIBLE_KEY, true);
    }

    public static function hideLogo() {
        self::init();
        self::$storage->set(self::LOGO_VISIBLE_KEY, false);
    }

    public static function isLogoVisible() {
        self::init();
        return self::$storage->get(self::LOGO_VISIBLE_KEY);
    }

    public static function setTitle($title) {
        self::init();
        self::$storage->set(self::TITLE_KEY, $title);
    }

    public static function getTitle() {
        self::init();
        return self::$storage->get(self::TITLE_KEY);
    }

    public static function setTemplate($template) {
        self::init();
        FileChecks::errorIfFileNotExists($template);
        self::$storage->set(self::TEMPLATE_KEY, $template);
    }

    public static function getTemplate() {
        self::init();
        return self::$storage->get(self::TEMPLATE_KEY);
    }

    public static function getMenuEntries()
    {
        self::init();
        $menu = self::$storage->get(self::MENU_KEY);
        if (!is_array($menu)) return array();
        return $menu;
    }

    public static function addMenuEntry($text, $template, $beforeExisting = true) {
        self::init();
        self::addMenuEntries(array($text => $template), $beforeExisting);
    }

    public static function removeMenuEntry($text) {
        self::init();
        $menu = self::getMenuEntries();;
        unset($menu[$text]);
        self::$storage->set(self::MENU_KEY, $menu);
    }

    public static function setMenuEntries(array $entries)
    {
        self::init();
        if (!$entries) {
            self::$storage->remove(self::MENU_KEY);
        } else {
            foreach ($entries as $entry) {
                FileChecks::errorIfFileNotExists($entry);
            }
            self::$storage->set(self::MENU_KEY, $entries);
        }
    }

    public static function addMenuEntries(array $entries, $beforeExisting = true) {
        self::init();
        $menu = array();
        if (!$beforeExisting) {
            foreach (self::getMenuEntries() as $text => $template) {
                if (!array_key_exists($text, $entries)) {
                    $menu[$text] = $template;
                }
            }
        }
        foreach ($entries as $text => $template) {
            FileChecks::errorIfFileNotExists($template);
            $menu[$text] = $template;
        }
        if ($beforeExisting) {
            foreach (self::getMenuEntries() as $text => $template) {
                if (!array_key_exists($text, $menu)) {
                    $menu[$text] = $template;
                }
            }
        }
        self::$storage->set(self::MENU_KEY, $menu);
    }

    public static function showMenu() {
        self::init();
        self::$storage->set(self::MENU_VISIBLE_KEY, true);
    }

    public static function hideMenu() {
        self::init();
        self::$storage->set(self::MENU_VISIBLE_KEY, false);
    }

    public static function isMenuVisible() {
        self::init();
        return self::$storage->get(self::MENU_VISIBLE_KEY);
    }

    public static function getMenuHtml() {
        self::init();
        return self::$storage->get(self::MENU_HTML_KEY);
    }

}
