<?php
/**
 * Konfiguration der Darstellung von Label/Value-Werten
 */
interface IfesLabelValueConfig
{
    function disable();
    function enable();
    /**
     * Labels ohne Mapping anzeigen (DEFAULT)
     *
     * @return IfesLabelValueConfig
     */
    function withUnmapped();

    /**
     * Labels ohne Mapping ausblenden
     *
     * @return IfesLabelValueConfig
     */
    function withoutUnmapped();

    /**
     * Standard-Mapping von Labels verwenden (DEFAULT)
     *
     * @return IfesLabelValueConfig
     */
    function withDefaultMapping();

    /**
     * Standard-Mapping von Labels ignorieren
     *
     * @return IfesLabelValueConfig
     */
    function withoutDefaultMapping();

    /**
     * Text von Warpit-Label durch eigenen Text ersetzen.
     * Die Reihenfolge der Aufrufe bestimmt die Reihenfolge der angezeigten Labels.
     *
     * @param string $warpitLabel Label-Text der ersetzt werden soll
     * @param string $newLabel (optional) Neuer Label-Text, wenn nicht angegeben
     *                         oder <true>, dann wird der Text aus dem Standard-Mapping
     *                         verwendet.
     * @return IfesLabelValueConfig
     */
    function map($warpitLabel, $newLabel = true);

    /**
     * Label nicht anzeigen, wenn der Wert in den angegebenen Filter-Werten enthalten ist
     *
     * @param string $warpitLabel Label-Text, dessen Werte gefiltert werden sollen
     * @param object $filterValues Werte, die gefiltert (nicht angezeigt) werden sollen
     *                             Anzahl der Parameter ist beliebig, z.B. filter('lbl', 1, 2, 3, 4, 5) ist ok.
     * @return IfesLabelValueConfig
     */
    function filter($warpitLabel, ...$filterValues);

    /**
     * Labels ohne Wert ausblenden
     *
     * @return IfesLabelValueConfig
     */
    function withoutEmptyValues();

    /**
     * Labels ohne Wert anzeigen (DEFAULT)
     *
     * @return IfesLabelValueConfig
     */
    function withEmptyValues();

    /**
     * Werte von Labels durch eigene Werte (Texte) ersetzen
     *
     * Beispiel für 'region':
     * $ifes->tStatus()->telInfo()->recode('region', array(
     *          1 => 'Vorarlberg',
     *          2 => 'Tirol',
     *          3 => 'Salzburg',
     *          4 => 'Oberösterreich',
     *          5 => 'Kärnten',
     *          6 => 'Steiermark',
     *          7 => 'Burgenland',
     *          8 => 'Niederösterreich',
     *          9 => 'Wien',
     *          10 => 'Mobil/RLD'));
     *
     * @param string $warpitLabel Label-Text, dessen Werte ersetzt werden sollen
     * @param array  $valueMapping Array mit neuen Werten (alterWert => neuerWert)
     * @return IfesLabelValueConfig
     */
    function recode($warpitLabel, $valueMapping);
}

interface IfesLabelValueConfigInternal extends IfesLabelValueConfig
{
    function process($items);
}

class IfesLabelValueConfigImpl implements IfesLabelValueConfigInternal
{
    private $enabled = true;
    private $skipUnmapped = false;
    private $ignoreDefaultMapping = false;
    private $mapping = array();
    private $valueFilter = array();
    private $recode = array();
    private $skipEmptyValues = false;

    private $defaultMapping;

    function __construct($defaultMapping)
    {
        $this->defaultMapping = $defaultMapping;
    }

    // start public interface

    public function enable()
    {
        $this->enabled = true;
        return $this;
    }

    public function disable()
    {
        $this->enabled = false;
        return $this;
    }

    public function withUnmapped()
    {
        $this->skipUnmapped = false;
        return $this;
    }

    public function withoutUnmapped()
    {
        $this->skipUnmapped = true;
        return $this;
    }

    public function withDefaultMapping()
    {
        $this->ignoreDefaultMapping = true;
        return $this;
    }

    public function withoutDefaultMapping()
    {
        $this->ignoreDefaultMapping = true;
        return $this;
    }

    public function withEmptyValues()
    {
        $this->skipEmptyValues = false;
        return $this;
    }

    public function withoutEmptyValues() {
        $this->skipEmptyValues = true;
        return $this;
    }

    public function map($warpitLabel, $newLabel = true)
    {
        if ($newLabel === true) {
            if (key_exists($warpitLabel, $this->defaultMapping)) {
                $newLabel = $this->defaultMapping[$warpitLabel];
            } else {
                $newLabel = $warpitLabel;
            }
        }

        $this->mapping[$warpitLabel] = $newLabel;
        return $this;
    }

    public function filter($warpitLabel, ...$filterValues)
    {
        if (!array_key_exists($warpitLabel, $this->valueFilter)) {
            $this->valueFilter[$warpitLabel] = array();
        }
        foreach ($filterValues as $value) {
            $this->valueFilter[$warpitLabel][] = strval($value);
        }
        return $this;
    }

    public function recode($warpitLabel, $valueMapping)
    {
        $this->recode[$warpitLabel] = $valueMapping;
        return $this;
    }

    // end public interface
    public function process($items)
    {
        if (!$this->enabled) return false;
        $mapping = $this->mapping;
        if (!$this->ignoreDefaultMapping) {
            foreach ($this->defaultMapping as $key => $value) {
                if (!array_key_exists($key, $mapping)) {
                    $mapping[$key] = $value;
                }
            }
        }

        $mappedItems = array();
        $mappedAndSortedItems = array();
        foreach ($items as $property => $value) {
            if ($this->skipUnmapped && !array_key_exists($property, $mapping)) continue;
            if ($this->filterValue($property, $value)) continue;
            if ($this->skipEmptyValues && (!$value || trim(strval($value)) == '')) continue;

            if (array_key_exists($property, $this->recode) && array_key_exists($value, $this->recode[$property])) {
                $value = $this->recode[$property][$value];
            }

            if (array_key_exists($property, $mapping)) {
                $property = $mapping[$property];
            }
            $mappedItems[$property] = $value;
        }

        foreach ($mapping as $value) {
            if (array_key_exists($value, $mappedItems)) {
                $mappedAndSortedItems[$value] = $mappedItems[$value];
            }
        }
        foreach ($mappedItems as $key => $value) {
            if (!array_key_exists($key, $mappedAndSortedItems)) {
                $mappedAndSortedItems[$key] = $value;
            }
        }
        return $mappedAndSortedItems;
    }

    private function filterValue($property, $value)
    {
        if (!array_key_exists($property, $this->valueFilter)) return false;
        return in_array(strval($value), $this->valueFilter[$property]);
    }
}

