<?php
include_once(IFES_ROOT_DIR . 'php/misc/IfesHelper.php');

/**
 * Konfiguration des Headers (Menü, Titel, Logo)
 */
interface IfesHeaderConfig
{
    /**
     * Header anzeigen (DEFAULT)
     * 
     * @return IfesHeaderConfig
     */
    function enable();

    /**
     * Header ausblenden
     * 
     * @return IfesHeaderConfig
     */
    function disable();

    /**
     * Logo anzeigen (DEFAULT)
     * 
     * @return IfesHeaderConfig
     */
    function withLogo();

    /**
     * Logo ausblenden
     * 
     * @return IfesHeaderConfig
     */
    function withoutLogo();

    /**
     * Titeltext im Header setzen
     * 
     * @param string $title Text der als Titel angezeigt werden soll
     * @return IfesHeaderConfig
     */
    function withTitle($title);

    /**
     * Titeltext ausblenden (DEFAULT)
     * 
     * @return IfesHeaderConfig
     */
    function withoutTitle();

    /**
     * Standard-Template für Header verwenden (DEFAULT)
     * 
     * @return IfesHeaderConfig
     */
    function withDefaultTemplate();

    /**
     * Spezielles-Template für Header verwenden
     * 
     * @param string $path Pfad zum Header-Template
     * @return IfesHeaderConfig
     */
    function withCustomTemplate($path);

    /**
     * Standard-Menü (Hamburger) anzeigen (DEFAULT)
     * 
     * @return IfesHeaderConfig
     */
    function withDefaultMenu();

    /**
     * Spezielles-Menü (Hamburger) anzeigen     * 
     * 
     * @param array $menuEntries Liste der Menü-Einträge
     *                           z.B. array('Menü-Eintrag' => 'pfad/zu/content.html', ...)
     * @return IfesHeaderConfig
     */
    function withCustomMenu($menuEntries);

    /**
     * Menü (Hamburger) nicht anzeigen
     * 
     * @return IfesHeaderConfig
     */
    function withoutMenu();
}

interface IfesHeaderConfigInternal extends IfesHeaderConfig
{
    function getHtml();
    function initVariables();
}

class IfesHeaderConfigImpl implements IfesHeaderConfigInternal
{
    /** @noinspection SpellCheckingInspection */
    const IFES_LOGO = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAL8AAABNCAYAAAD3sLUUAAAABHNCSVQICAgIfAhkiAAAAjRJREFU eJzt3cFRg1AUheGjkwJSAiXYgWZPD2bHOhUkVuCaXeyBvSmBErADSnDBJnGiicmD+5Lzf0uG4V30 X+AwPB9U1nNJT0qrU1N1J88q60JSkXjtKfRqqvbgyH3dyxhN5KadabjJz8QXfpO0OeO8paR14rWn sJO0+HFsqfu5lzGayM3iMXoCIArxwxbxwxbxwxbxwxbxwxbxw9YseoAjWkmr6CFO6APX3kr6SHi9 a+9lpeF3dmvaHOPv1VS76CEy9pXZz6fNbJ6z8dgDW8QPW8QPW8QPW8QPW8QPW8QPW8QPW8QPW8QP W8QPW8QPW8QPW8QPW8QPW8QPW8QPW8QPWzl+xvikss59n8hWTRX1nfGryvo54fWuvZd3lXXkN82X WuUY/1zSS/QQGSuU127Qt7qb85zHHtgiftgiftgiftgiftgiftgiftgiftjK8SWXg7foAfZ0vxyL mvFZE73kJP4ITbWJHuFPw/9Q3oSsXdYbEf/FtpK+Rl6jG/n6mMA9xv9xq/vFY1r8wQtbxA9bxA9b xA9bxA9bxA9bxA9bxA9bxA9bxA9bxA9bxA9bxA9bxA9bxA9bxA9bxA9bxA9bxA9bxA9bxA9b97h7 Q/6GvWly0amptgdHyrqQtAyYRRo2rZoE8cdYRw+wZ6dhr6N9hfKacRQ89sAW8cMW8cMW8cMW8cMW 8cMW8cMW8cMW8cMW8cPWTFKv4RV3St0/zku9dp/4eufqlP5eptAeOTZGE7npvwFzyVALhFGDUQAA AABJRU5ErkJggg==';
    /** @noinspection SpellCheckingInspection */
    const IFES_LOGO_VERTICAL = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAE0AAAC/CAYAAAC7U1GiAAAAAXNSR0IArs4c6QAAA/ZJREFUeF7t3T1OG0EchvH/GIqUPgJH4AYhbWQkbhDSEFFygiQnoLSgAW6AFJQW5wY+ArmBSwrijXYjvqTYmXnYmLX9uM2+Y8/P7zr7TYq9s37c3W7HvNfmm3FcfpzMXabkH9+fbkfvV78k0pllpxuTFIPhTkTvev6Hmr6Lq8NRax98cHodUe20Nt5CB0oj0YrBRSsmixBNNCAAIjZNNCAAIjZNNCAAIjZNNCAAIjZNNCAAIjZNNCAAIjZNNCAAIjZNNCAAIjZNNCAAIjZNNCAAIjZNNCAAIp1vWhpFVD/AzAoi6W3ZievOo8XXuPr0pUCgfNHBST3+5/ygaBGi5fflYUnRRKsF/E17XCWyL7USTTTw8yGaaJ29fNTfNFdPV09Xz/wOuO/pvmd+W54s6b4nYBNNNHfYn3Ug/96oJd64reIo0nQMyj8j0juOiPl3/v1JLjFae1qlI4lWKmbTgJhoojEBkPI3TTQgACI2TTQgACI2bXXQch5mAmbbSmQzbuLy8KaVsWYNsjfcirvYyn6P5mEmvooFRCsmixBNNCAAIimaJ0xFfXxr9msaR/H9oL3jabsnx1Glfx9PS3ER3w7OwbzyI7un+1HFh+xAqsZdf0DTMm+nvdqjwER7WAXW4hxB2LTH3zwvS/CyhOz/AZ8u6Bl2wCaaaLWAmxxucoA1QTTRvOY2vwNu3Lpxm9+WJ0u6nQbYRBPNjdtnHViL42lVdR4p/QTl/3ukPiafsk7QLvNuVGtcpQOJViq25DvsYLrtRGwacBRNNCAAIjZNNCAAIjZNNCAAIjZNNCAAIjZNNCAAIjZNNCCwGpHsU3irMd12ZiEacBRNNCAAIjZNNCAAIjZNNCAAIjYNojW3I/f2QXo9IyndeJMs+OpFEw0IgIhNQ2hdvh37fkKLuC27AK/rt2PfT+X/H8EVrUAALGrTRAMCIGLTRAMCIGLTRAMCIGLTRAMCIGLTRAMCIGLTRAMCIGLTRAMCIGLTRAMCIGLTRAMCIGLTRAMCIGLTRAMCILIcTWueC1JdgPm1H2n+iMNguBPR2eentT/pF4+YfVHfqz1R+cVTbH8A0YCpaKIBARCxaaIBARCxaaIBARCxaaIBARCxaaIBARCxaaIBARDJblqMI9IEvMOMSFX/Hbx+e+MtcqR8tEV+qo6/l2jgCxJNNCAAInXTvB27DM7bscu87pf2JlngJppoQABEbJpoQABEbJpoQABEbBpCc4+gjK3ZI8i61Kps3NVe2qMc4PsVTTQgACI2TTQgACI2TTQgACI2TTQgACI2TTQgACI2TTQgACI2TTQgACI2TTQgACI2TTQgACLNpVZn/bi7ra+B9ZUjMN2Y/Ab6nJ2G00QRkQAAAABJRU5ErkJggg==';
    const DEFAULT_TEMPLATE_FILE = IFES_ROOT_DIR . 'templates/header.html';
    const DEFAULT_LOGO_ALT_TEXT = 'IFES Logo';
    const DEFAULT_MENU_ENTRIES = [
        'Hilfe' => IFES_ROOT_DIR . 'templates/default-menu/help.html',
        'Impressum' => IFES_ROOT_DIR . 'templates/default-menu/impressum.html'
    ];
    const BUTTON_HTML = '<button type="button" class="navbar-toggler"><div class="hamburger"><span></span><span></span><span></span></div></button>';
    const NO_BUTTON_HTML = '<div class="navbar-toggler"></div>';

    const VAR_LOGO = 'header_logo';
    const VAR_TITLE = 'header_title';
    const VAR_MENU = 'header_menu';
    const VAR_MENU_BUTTON = 'header_menu_button';
    const VAR_YEAR = 'year';

    private $enabled;
    private $logo;
    private $title;
    private $template_file;
    private $menu;

    function __construct()
    {
        $this->enabled = true;
        $this->logo = true;
        $this->title = "";
        $this->menu = self::DEFAULT_MENU_ENTRIES;
        $this->template_file = self::DEFAULT_TEMPLATE_FILE;
    }

    public function enable()
    {
        $this->enabled = true;
        return $this;
    }

    public function disable()
    {
        $this->enabled = false;
        return $this;
    }

    public function withLogo()
    {
        $this->logo = true;
        return $this;
    }

    public function withoutLogo()
    {
        $this->logo = false;
        return $this;
    }

    public function withTitle($title)
    {
        $this->title = $title;
        return $this;
    }

    public function withoutTitle()
    {
        $this->title = false;
        return $this;
    }

    public function withDefaultTemplate()
    {
        $this->template_file = self::DEFAULT_TEMPLATE_FILE;
        return $this;
    }

    public function withCustomTemplate($path)
    {
        $this->template_file = $path;
        return $this;
    }

    public function withDefaultMenu()
    {
        $this->menu = self::DEFAULT_MENU_ENTRIES;
        return $this;
    }

    public function withCustomMenu($menuEntries)
    {
        $this->menu = $menuEntries;
        return $this;
    }

    public function withoutMenu()
    {
        $this->menu = false;
        return $this;
    }

    public function initVariables()
    {
        // do nothing
    }

    public function getHtml()
    {
        if (!$this->enabled) return '';

        $vars = array(
            self::VAR_LOGO => $this->getLogoHtml(),
            self::VAR_TITLE => $this->getTitleText(),
            self::VAR_MENU => $this->getMenuEntriesHtml(),
            self::VAR_MENU_BUTTON => ($this->menu ? self::BUTTON_HTML : self::NO_BUTTON_HTML),
            self::VAR_YEAR => date("Y")
        );
        $template = $this->getTemplate();

        return IfesHelper::replaceVars($template, $vars);
    }

    private function getMenuEntriesHtml()
    {
        $menu = $this->menu;
        $html = '';
        $content = '';
        $linkCount = 0;
        if (is_array($menu) && count($menu) > 0) {
            $html = "<div class=\"navbar-links\">\n<ul class=\"navbar-links\">\n";
            foreach ($menu as $link => $template) {
                /** @noinspection PhpStrictComparisonWithOperandsOfDifferentTypesInspection */
                if ($template === true && array_key_exists($link, self::DEFAULT_MENU_ENTRIES)) {
                    $template = self::DEFAULT_MENU_ENTRIES[$link];
                }
                /** @noinspection PhpStrictComparisonWithOperandsOfDifferentTypesInspection */
                if ($template === false) continue;
                $linkCount++;
                $id = "menu-entry-$linkCount";
                $html .= "<li class=\"navbar-link\"><a class=\"nav-link\" id=\"$id-link\" href=\"#\" onclick=\"showMenuEntry('#$id');\">$link</a></li>\n";

                if (!file_exists($template)) {
                    $content .= "<div id=\"$id\" class=\"navbar-menu-content\">Menu template '$template' not found!</div>\n";
                } else {
                    $content .= "<div id=\"$id\" class=\"navbar-menu-content\">\n" . file_get_contents($template) . "\n</div>\n";
                }
            }
            $html .= "</ul>\n</div>\n";
            $html .= "<div class=\"navbar-content\">\n";
            $html .= $content;
            $html .= "</div>\n";
        }
        return $html;
    }

    private function getLogoHtml()
    {
        $html = '';
        if ($this->logo) {
            $html = '<img class="ifes-logo" alt="' . self::DEFAULT_LOGO_ALT_TEXT . '" src="' . self::IFES_LOGO . '" />';
            $html .= '<img class="ifes-logo vertical" alt="' . self::DEFAULT_LOGO_ALT_TEXT . '" src="' . self::IFES_LOGO_VERTICAL . '" />';
        }
        return $html;
    }

    private function getTitleText()
    {
        $title = '';
        if (is_string($this->title)) $title = $this->title;
        return htmlspecialchars($title, ENT_QUOTES);
    }

    private function getTemplate()
    {
        $template = $this->template_file;

        if (!file_exists($template)) {
            return "<div>Header template '$template' not found!</div>";
        } else {
            return file_get_contents($template);
        }
    }
}
