<?php
include_once(IFES_ROOT_DIR . 'php/misc/IfesHelper.php');
include_once(IFES_ROOT_DIR . 'php/config/IfesProgressbarConfig.php');
include_once(IFES_ROOT_DIR . 'php/config/IfesHeaderConfig.php');
include_once(IFES_ROOT_DIR . 'php/config/IfesButtonConfig.php');
include_once(IFES_ROOT_DIR . 'php/config/IfesTStatusConfig.php');
include_once(IFES_ROOT_DIR . 'php/config/IfesAppointmentConfig.php');
include_once(IFES_ROOT_DIR . 'php/config/IfesWarpitInfo.php');

/**
 * Root-Objekt für die Konfiguration.
 * 
 * Ein initialisiertes Objekt von IfesConfig kann in allen Scripts
 * über die Variable $ifes angesprochen werden.
 */
interface IfesConfig
{
    /**
     * Aktiviert die Ausgabe von PHP Fehlermeldungen.
     * Hilfreich beim Entwickeln. 
     * 
     * Beim Verwenden der Funktion wird im HTML eine Warnung angezeigt,
     * dass die Funktion aktiv ist.
     * 
     * Muss bei Produktiv-Projekten deaktiviert werden!!!
     * 
     * @return void
     */
    function displayPhpErrors();

    /**
     * Schreibt eine Variable mit 'print_r' inkl. HTML Pre-Tag.
     * 
     * @param object $var Ein PHP Wert/Variable/...
     * @return void
     */
    function printVar($var);

    /**
     * Schreibt eine Variable mit 'var_dump' inkl. HTML Pre-Tag.
     * 
     * @param object $var Ein PHP Wert/Variable/...
     * @return void
     */
    function dumpVar($var);

    /**
     * Schreibt ein HTML Script-Tag (JavaScript)
     * 
     * @param string $path Pfad zur JavaScript-Datei
     * @return IfesConfig
     */
    function addJavaScript($path);

    /**
     * Schreibt ein HTML Link-Tag (CSS)
     * 
     * @param string $path Pfad zur CSS-Datei
     * @return IfesConfig
     */
    function addStylesheet($path);

    /**
     * Pfad zum Projekt-Ordner für zusätzliche Dateien
     * 
     * Standard ist PATH_TO_ROOT/../webIfesProjects/ProjFiles/[Projektnummer]
     * Kann durch einen Kommentar im Projekt-Script 'Start of survey' konfiguriert werden.
     * 
     * @return string Relativer Pfad zum Ordner
     */
    function projectDir();

    /**
     * Liefert den Pfad zum Laden von Dateien im Projekt-Ordner
     * 
     * @param string $relativePath Relativer Pfad innerhalb des Projekt-Ordners
     * @return string {ProjectDir}/{$relativePath}
     */
    function toProjectPath($relativePath);

    /**
     * Progressbar Konfiguration
     * 
     * @return IfesProgressbarConfig
     */
    function progressbar();

    /**
     * Header Konfiguration
     * 
     * @return IfesHeaderConfig
     */
    function header();

    /**
     * tStatus Konfiguration (CATI Startseite)
     * 
     * @return IfesTStatusConfig
     */
    function tStatus();

    /**
     * Button Konfiguration
     * 
     * @return IfesButtonConfig
     */
    function buttons();

    function appointment();

    function warpit();

    function enableTestInterviewMsg($message);

    function disableTestInterviewMsg();


    function nextQuestion();

}

interface IfesConfigInternal extends IfesConfig {
    function initVariables();
    function getCssFiles();
    function getJsFiles();
    function validateApiVersion();
    function getTestMsg();
}

class IfesConfigImpl implements IfesConfigInternal
{
    CONST API_VERSIONS = ['v1'];
    CONST DEFAULT_TEST_INTERVIEW_MESSAGE = "Test-Interview";

    CONST CATI_SUPERVISOR_GROUP = 'CATI-SVs';

    private $db;
    private $questionnaire;
    private $projectDir;
    private $cssDir;
    private $jsDir;

    private $progressbar;
    private $header;
    private $buttons;
    private $tStatus;
    private $appointment;
    private $warpit;
    private $testMsg = self::DEFAULT_TEST_INTERVIEW_MESSAGE;
    private $css;
    private $js;
    private $vars = array();
    private $apiVersion;

    function __construct($db, $questionnaire)
    {
        $this->db = $db;
        $this->questionnaire = $questionnaire;

        $this->loadStaticConfig();

        $this->progressbar = new IfesProgressbarConfigImpl($db, $questionnaire);
        $this->header = new IfesHeaderConfigImpl();
        $this->buttons = new IfesButtonConfigImpl();
        $this->tStatus = new IfesTStatusConfigImpl();
        $this->appointment = new IfesAppointmentConfigImpl();
        $this->warpit = new IfesWarpitInfoImpl($db, $questionnaire);

        if (!$this->projectDir) {
            $projectDir = $this->questionnaire->projname;
            if (preg_match('/[^0-9]?([0-9]*).*/i', $projectDir, $matches)) {
                $projectDir = $matches[1];
            }
            $this->projectDir = IFES_PROJECT_ROOT_DIR . $projectDir . '/';
        }
        if (substr($this->projectDir, -1) !== '/') $this->projectDir .= '/';

        $this->css = array(
            IFES_ROOT_DIR . 'css/ifes.css',
            IFES_ROOT_DIR . 'css/header.css'
        );
        $this->js = array();

        if (!$this->cssDir) $this->cssDir = $this->projectDir . 'css';
        $this->getAdditionalCssFiles();
        if (!$this->jsDir) $this->jsDir = $this->projectDir . 'js';
        $this->getAdditionalJsFiles();
    }

    public function nextQuestion()
    {
        if ($_SESSION['nextQuestion']) {
            return $_SESSION['nextQuestion'][0]['q_name'];
        } else {
            return '';
        }
    }

    public function projectDir() {
        return $this->projectDir;
    }

    public function toProjectPath($relativePath) {
        return $this->projectDir() . trim($relativePath);
    }

    public function progressbar()
    {
        return $this->progressbar;
    }

    public function header() {
        return $this->header;
    }

    public function tStatus() {
        return $this->tStatus;
    }

    public function buttons() {
        return $this->buttons;
    }

    public function appointment()
    {
        return $this->appointment;
    }

    function warpit()
    {
        return $this->warpit;
    }

    public function addJavaScript($path)
    {
        echo "<script type=\"text/javascript\" src=\"$path\"></script>";
        return $this;
    }

    public function addStylesheet($path)
    {
        echo "<link rel=\"stylesheet\" type=\"text/css\" href=\"$path\" />";
        return $this;
    }

    public function enableTestInterviewMsg($message = '')
    {
        if (!$message) $message = self::DEFAULT_TEST_INTERVIEW_MESSAGE;
        $this->testMsg = trim($message);
    }

    public function disableTestInterviewMsg()
    {
        $this->testMsg = false;
    }

    public function displayPhpErrors()
    {
        ini_set('display_errors', 1);
        error_reporting(E_ERROR | E_WARNING | E_PARSE | E_USER_ERROR | E_USER_WARNING);
        $this->showWarning('IFES Design :: Display PHP Errors ist aktiv.');
        }

    public function printVar($var)
    {
        IfesHelper::printVar($var);
    }

    public function dumpVar($var)
    {
        IfesHelper::dumpVar($var);
    }

    public function initVariables()
    {
        $this->buttons->initVariables();
        $this->progressbar->initVariables();
        $this->header->initVariables();

        foreach ($this->vars as $key => $value) {
            IfesHelper::setDesignerVariable($key, $value);
        }
    }

    public function getCssFiles()
    {
        return $this->css;
    }

    public function getJsFiles()
    {
        return $this->js;
    }

    public function getTestMsg()
    {
        return $this->testMsg;
    }

    public function validateApiVersion() {
        if (!$this->apiVersion) {
            $this->showWarning('IFES Design :: API Version muss noch konfiguriert werden.');
        } else if (!in_array($this->apiVersion, self::API_VERSIONS)) {
            $this->showWarning("IFES Design :: API Version existiert nicht (Falscher Wert: $this->apiVersion)");
        }
    }

    private function loadStaticConfig()
    {
        $table = DB_WARPIT_WEBCATI . '.' . $this->questionnaire->projname . '_script ';
        $where = 'id_question = 0 AND id_event = 3';
        $script = $this->db->get($table, $where, 'script');
        if (!$script) return;

        $script = reset($script);
        $config = array();
        $isConfig = false;
        foreach (preg_split("/((\r?\n)|(\n?\r))/", $script) as $line) {
            $line = trim($line);
            if (empty($line)) continue;
            if (str_replace(' ', '', $line) == '/*IFES_CONFIG_START') {
                $isConfig = true;
                continue;
            }
            if (str_replace(' ', '', $line) == 'IFES_CONGIG_END*/') break;
            if (!$isConfig) continue;
            if (!str_contains($line, "=")) continue;
            list($property, $value) = explode('=', $line);
            $config[strtolower(trim($property))] = trim($value);
        }

        foreach ($config as $property => $value) {
            switch ($property) {
                case 'api_version':
                    $this->apiVersion = $value;
                    break;

                case 'project_dir':
                    $this->projectDir = $this->setProjectDirPath(trim($value));
                    break;
            }
        }
    }

    private function setProjectDirPath($dir) {
        if (substr($dir, 1) != '.') {
            $dir = IFES_PROJECT_ROOT_DIR . $dir . '/';
        }

        if (!is_dir($dir)) {
            trigger_error("IFES CONFIG WARNING: Directory $dir does not exist", E_USER_WARNING);
        }

        return $dir;
    }

    private function getAdditionalCssFiles() {
        $dir = $this->projectDir . 'css/';
        $this->css = array_merge($this->css, $this->getFiles($dir, '.css'));
        }

    private function getAdditionalJsFiles() {
        $dir = $this->projectDir . 'js/';
        $this->js = array_merge($this->js, $this->getFiles($dir, '.js'));
    }

    private function getFiles($dir, $fileExtension) {
        if (!is_dir($dir)) return array();

        $files = array();
        foreach(scandir($dir, SCANDIR_SORT_ASCENDING) as $file) {
            if (substr($file, strlen($fileExtension) * -1) == $fileExtension) {
                $files[] = $dir . $file;
            }
        }
        return $files;
    }

    private function showWarning($text) {
        echo "<div class=\"ifes-dev-warning\">$text</div>";
    }
}
