<?php
include_once(IFES_ROOT_DIR . 'php/misc/IfesHelper.php');

/**
 * Konfiguration der Buttons (Weiter, Zurück)
 */
interface IfesButtonConfig
{
    /**
     * Text für Weiter-Button ändern (DEFAULT: Weiter)
     * 
     * @param string $text Text für Button
     * @return IfesButtonConfig
     */
    function withNextButtonText($text);

    /**
     * Text für Zurück-Button ändern (DEFAULT: Zurück)
     * 
     * @param string $text Text für Button
     * @return IfesButtonConfig
     */
    function withBackButtonText($text);

    /**
     * Text für Hangup-Button ändern (DEFAULT: Auflegen)
     *
     * @param string $text Text für Button
     * @return IfesButtonConfig
     */
    function withHangupButtonText($text);

    /**
     * Zurück-Button anzeigen (DEFAULT)
     * 
     * @return IfesButtonConfig
     */
    function showBackButton();

    /**
     * Zurück-Button ausblenden
     * 
     * @return IfesButtonConfig
     */
    function hideBackButton();

    /**
     * Hangup-Button anzeigen
     *
     * @return IfesButtonConfig
     */
    function showHangupButton();

    /**
     * Hangup-Button ausblenden (DEFAULT)
     *
     * @return IfesButtonConfig
     */
    function hideHangupButton();
}

interface IfesButtonConfigInternal extends IfesButtonConfig
{
    function isBackButtonHidden();
    function getSubmitBtnText();
    function getBackBtnText();
    function initVariables();
    function isHangupButtonHidden();
    function getHangupButtonText();
}

class IfesButtonConfigImpl implements IfesButtonConfigInternal
{
    const DEFAULT_SUBMIT_BTN_TEXT = 'Weiter';
    const DEFAULT_BACK_BTN_TEXT = 'Zurück';
    const DEFAULT_HANGUP_BTN_TEXT = 'Auflegen';
    const VAR_BACK_BTN_TEXT = 'back_button_text';
    const VAR_SUBMIT_BTN_TEXT = 'submit_button_text';
    const VAR_BACK_BTN_VISIBILITY = 'back_button_visibility';

    private $nextButtonText;
    private $backButtonText;
    private $showBackButton;

    private $hangupButtonText;
    private $showHangupButton;

    function __construct()
    {
        $this->nextButtonText = self::DEFAULT_SUBMIT_BTN_TEXT;
        $this->backButtonText = self::DEFAULT_BACK_BTN_TEXT;
        $this->showBackButton = true;
        $this->hangupButtonText = self::DEFAULT_HANGUP_BTN_TEXT;
        $this->showHangupButton = false;
    }

    public function withNextButtonText($text)
    {
        $this->nextButtonText = $text;
        return $this;
    }

    public function withBackButtonText($text)
    {
        $this->backButtonText = $text;
        return $this;
    }

    function withHangupButtonText($text)
    {
        $this->hangupButtonText = $text;
        return $this;
    }

    function showHangupButton()
    {
        $this->showHangupButton = true;
        return $this;
    }

    function hideHangupButton()
    {
        $this->showHangupButton = false;
        return $this;
    }

    public function showBackButton()
    {
        $this->showBackButton = true;
        return $this;
    }

    public function hideBackButton()
    {
        $this->showBackButton = false;
        return $this;
    }

    public function isBackButtonHidden()
    {
        return !$this->showBackButton;
    }

    function isHangupButtonHidden()
    {
        return !$this->showHangupButton;
    }

    public function getSubmitBtnText()
    {
        return htmlspecialchars($this->nextButtonText, ENT_QUOTES);
    }

    public function getBackBtnText()
    {
        return htmlspecialchars($this->backButtonText, ENT_QUOTES);
    }

    function getHangupButtonText()
    {
        return htmlspecialchars($this->hangupButtonText, ENT_QUOTES);
    }

    public function initVariables()
    {
        IfesHelper::setDesignerVariable(self::VAR_BACK_BTN_TEXT, $this->getBackBtnText());
        IfesHelper::setDesignerVariable(self::VAR_BACK_BTN_VISIBILITY, $this->showBackButton ? 'block' : 'hidden');
        IfesHelper::setDesignerVariable(self::VAR_SUBMIT_BTN_TEXT, $this->getSubmitBtnText());
    }
}